<!DOCTYPE html>
<html lang="hr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>TapTap Business Card - Admin Section</title>
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    
    body {
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      background: #f5f7fa;
    }
    
    .login-container {
      display: flex;
      justify-content: center;
      align-items: center;
      min-height: 100vh;
      padding: 20px;
    }
    
    .login-box {
      background: white;
      padding: 40px;
      border-radius: 12px;
      box-shadow: 0 4px 20px rgba(0,0,0,0.1);
      width: 100%;
      max-width: 400px;
    }
    
    .login-box h2 { margin-bottom: 30px; color: #333; text-align: center; }
    .form-group { margin-bottom: 20px; }
    
    label {
      display: block;
      margin-bottom: 8px;
      font-weight: 600;
      color: #555;
      font-size: 14px;
    }
    
    input, textarea, select {
      width: 100%;
      padding: 12px;
      border: 2px solid #e5e7eb;
      border-radius: 8px;
      font-size: 15px;
      transition: border 0.3s;
      font-family: inherit;
    }
    
    input:focus, textarea:focus, select:focus {
      outline: none;
      border-color: #667eea;
    }
    
    textarea { resize: vertical; min-height: 100px; }
    
    .btn {
      width: 100%;
      padding: 14px;
      background: #667eea;
      color: white;
      border: none;
      border-radius: 8px;
      font-size: 16px;
      font-weight: 600;
      cursor: pointer;
      transition: background 0.3s;
    }
    
    .btn:hover { background: #5568d3; }
    .btn-secondary { background: #6b7280; }
    .btn-secondary:hover { background: #4b5563; }
    .btn-danger { background: #ef4444; }
    .btn-danger:hover { background: #dc2626; }
    .btn-small { width: auto; padding: 8px 16px; font-size: 14px; }
    
    .dashboard {
      max-width: 1400px;
      margin: 0 auto;
      padding: 20px;
    }
    
    .header {
      background: white;
      padding: 20px;
      border-radius: 12px;
      margin-bottom: 30px;
      display: flex;
      justify-content: space-between;
      align-items: center;
      box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    .header h1 { color: #333; }
    
    .tabs {
      display: flex;
      gap: 10px;
      margin-bottom: 20px;
      border-bottom: 2px solid #e5e7eb;
    }
    
    .tab {
      padding: 12px 20px;
      background: none;
      border: none;
      border-bottom: 3px solid transparent;
      cursor: pointer;
      font-size: 15px;
      font-weight: 600;
      color: #6b7280;
      transition: all 0.3s;
    }
    
    .tab:hover {
      color: #667eea;
    }
    
    .tab.active {
      color: #667eea;
      border-bottom-color: #667eea;
    }
    
    .tab-content {
      display: none;
    }
    
    .tab-content.active {
      display: block;
    }
    
    .users-grid { display: grid; gap: 20px; }
    
    .user-card {
      background: white;
      padding: 25px;
      border-radius: 12px;
      box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    .user-header {
      display: flex;
      justify-content: space-between;
      align-items: start;
      margin-bottom: 20px;
    }
    
    .user-info h3 { color: #333; margin-bottom: 5px; }
    .user-slug { color: #667eea; font-size: 14px; }
    .user-actions { display: flex; gap: 10px; }
    
    .stats {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
      gap: 15px;
      margin-top: 20px;
      padding-top: 20px;
      border-top: 1px solid #e5e7eb;
    }
    
    .stat-box {
      text-align: center;
      padding: 15px;
      background: #f8f9ff;
      border-radius: 8px;
    }
    
    .stat-number {
      font-size: 28px;
      font-weight: 700;
      color: #667eea;
    }
    
    .stat-label {
      font-size: 13px;
      color: #6b7280;
      margin-top: 5px;
    }
    
    .modal {
      display: none;
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0,0,0,0.5);
      z-index: 1000;
      padding: 20px;
      overflow-y: auto;
    }
    
    .modal.active {
      display: flex;
      justify-content: center;
      align-items: start;
      padding-top: 50px;
    }
    
    .modal-content {
      background: white;
      padding: 30px;
      border-radius: 12px;
      width: 100%;
      max-width: 900px;
      max-height: 90vh;
      overflow-y: auto;
    }
    
    .modal-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 25px;
      padding-bottom: 15px;
      border-bottom: 2px solid #e5e7eb;
    }
    
    .modal-header h2 { color: #333; }
    
    .close-btn {
      background: none;
      border: none;
      font-size: 28px;
      cursor: pointer;
      color: #6b7280;
      line-height: 1;
    }
    
    .form-row {
      display: grid;
      grid-template-columns: 1fr 1fr;
      gap: 20px;
    }
    
    .form-section {
      margin-top: 30px;
      padding-top: 30px;
      border-top: 2px solid #f3f4f6;
    }
    
    .form-section h3 {
      color: #333;
      margin-bottom: 20px;
      font-size: 18px;
    }
    
    .template-grid, .background-grid {
      display: grid;
      grid-template-columns: repeat(4, 1fr);
      gap: 15px;
      margin-top: 15px;
    }
    
    .template-option, .background-option {
      position: relative;
      border: 3px solid #e5e7eb;
      border-radius: 12px;
      padding: 20px;
      text-align: center;
      cursor: pointer;
      transition: all 0.3s;
    }
    
    .template-option:hover, .background-option:hover {
      border-color: #667eea;
      transform: translateY(-2px);
    }
    
    .template-option.active, .background-option.active {
      border-color: #667eea;
      background: #f8f9ff;
    }
    
    .template-option input, .background-option input {
      position: absolute;
      opacity: 0;
      width: 0;
      height: 0;
    }
    
    .template-preview {
      width: 100%;
      height: 80px;
      border-radius: 8px;
      margin-bottom: 10px;
    }
    
    .background-preview {
      width: 100%;
      height: 60px;
      border-radius: 8px;
      margin-bottom: 10px;
    }
    
    .social-links-editor {
      display: grid;
      gap: 15px;
    }
    
    .link-row {
      display: grid;
      grid-template-columns: 150px 1fr 50px;
      gap: 10px;
      align-items: center;
      padding: 15px;
      background: #f8f9fa;
      border-radius: 8px;
    }
    
    .link-row select {
      padding: 10px;
    }
    
    .remove-link-btn {
      background: #ef4444;
      color: white;
      border: none;
      border-radius: 6px;
      padding: 10px;
      cursor: pointer;
      font-size: 18px;
      line-height: 1;
    }
    
    .add-link-btn {
      margin-top: 15px;
      background: #10b981;
      color: white;
      border: none;
      border-radius: 8px;
      padding: 12px;
      cursor: pointer;
      font-weight: 600;
      width: 100%;
    }
    
    .add-link-btn:hover {
      background: #059669;
    }
    
    .error-message, .success-message {
      padding: 12px;
      border-radius: 8px;
      margin-bottom: 15px;
    }
    
    .error-message { background: #fee; color: #c00; }
    .success-message { background: #efe; color: #0a0; }
    
    .checkbox-grid {
      display: grid;
      grid-template-columns: repeat(2, 1fr);
      gap: 15px;
    }
    
    .checkbox-item {
      display: flex;
      align-items: center;
      gap: 10px;
      padding: 12px;
      background: #f8f9fa;
      border-radius: 8px;
      cursor: pointer;
    }
    
    .checkbox-item input[type="checkbox"] {
      width: auto;
      cursor: pointer;
    }
    
    .checkbox-item label {
      margin: 0;
      cursor: pointer;
    }
    
    .card {
      background: white;
      padding: 30px;
      border-radius: 12px;
      box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    @media (max-width: 768px) {
      .header { flex-direction: column; gap: 15px; }
      .user-header { flex-direction: column; gap: 15px; }
      .form-row { grid-template-columns: 1fr; }
      .template-grid, .background-grid { grid-template-columns: repeat(2, 1fr); }
      .link-row { grid-template-columns: 1fr; }
    }
  </style>
</head>
<body>
  <!-- Login View -->
  <div id="loginView" class="login-container">
    <div class="login-box">
      <h2>🔒 Admin Prijava</h2>
      <div id="loginError" class="error-message" style="display: none;"></div>
      <div class="form-group">
        <label>Korisničko ime</label>
        <input type="text" id="loginUsername" placeholder="admin" value="admin">
      </div>
      <div class="form-group">
        <label>Lozinka</label>
        <input type="password" id="loginPassword" placeholder="••••••••">
      </div>
      <button class="btn" onclick="handleLogin()">Prijavi se</button>
    </div>
  </div>

  <!-- Dashboard View -->
  <div id="dashboardView" class="dashboard" style="display: none;">
    <div class="header">
      <h1>📇 TapTap Card Admin</h1>
      <div style="display: flex; gap: 10px; flex-wrap: wrap;">
  <button class="btn btn-small" onclick="openCreateModal()">+ Novi korisnik</button>
  <button class="btn btn-small" style="background: #10b981;" onclick="exportAll()">⬇️ Export sve</button>
  <button class="btn btn-small" style="background: #f59e0b;" onclick="document.getElementById('importFile').click()">⬆️ Import</button>
  <input type="file" id="importFile" accept=".json" style="display: none;" onchange="handleImport(event)">
  <button class="btn btn-small btn-secondary" onclick="handleLogout()">Odjava</button>
</div>
    </div>
    
    <div class="tabs">
      <button class="tab active" onclick="showTab('users')">👥 Korisnici</button>
      <button class="tab" onclick="showTab('settings')">⚙️ Settings</button>
    </div>
    
    <div id="users-tab" class="tab-content active">
      <div id="usersList" class="users-grid"></div>
    </div>
    
    <div id="settings-tab" class="tab-content">
      <div class="card" style="max-width: 600px; margin: 0 auto;">
        <h2 style="margin-bottom: 20px;">🌐 Platform Settings</h2>
        
        <div class="form-group">
          <label>Analytics Slug</label>
          <input type="text" id="settingAnalyticsSlug" placeholder="analitika">
          <small style="color: #64748b; font-size: 12px; margin-top: 5px; display: block;">
            Link: /<strong id="previewSlug">analitika</strong><br>
            Spremi za višejezičnost (npr: analytics, analytik...)
          </small>
        </div>
        
        <button onclick="saveSettings()" class="btn" style="width: 100%;">
          💾 Spremi postavke
        </button>
      </div>
    </div>
  </div>

  <!-- Edit Modal -->
  <div id="userModal" class="modal">
    <div class="modal-content">
      <div class="modal-header">
        <h2 id="modalTitle">Novi korisnik</h2>
        <button class="close-btn" onclick="closeModal()">×</button>
      </div>
      
      <div id="modalError" class="error-message" style="display: none;"></div>
      
      <form id="userForm" onsubmit="handleSubmit(event)">
        <input type="hidden" id="userId">
        
        <!-- Osnovni podaci -->
        <div class="form-row">
          <div class="form-group">
            <label>Slug (URL)*</label>
            <input type="text" id="slug" required placeholder="npr. zmadzar">
          </div>
          <div class="form-group">
            <label>Profilna slika</label>
            <input type="file" id="slika" accept="image/*">
          </div>
        </div>
        
        <div class="form-row">
          <div class="form-group">
            <label>Ime*</label>
            <input type="text" id="ime" required>
          </div>
          <div class="form-group">
            <label>Prezime*</label>
            <input type="text" id="prezime" required>
          </div>
        </div>
        
        <div class="form-group">
          <label>Opis</label>
          <textarea id="opis" placeholder="Kratki opis o sebi..."></textarea>
        </div>
        
        <!-- Kontakt podaci -->
        <div class="form-section">
          <h3>📞 Kontakt podaci</h3>
          <div class="form-row">
            <div class="form-group">
              <label>Email</label>
              <input type="email" id="email">
            </div>
            <div class="form-group">
              <label>Telefon</label>
              <input type="tel" id="telefon">
            </div>
          </div>
          <div class="form-row">
            <div class="form-group">
              <label>WhatsApp</label>
              <input type="tel" id="whatsapp">
            </div>
            <div class="form-group">
              <label>Viber</label>
              <input type="tel" id="viber">
            </div>
          </div>
        </div>
        
        <!-- Adresa -->
        <div class="form-section">
          <h3>📍 Adresa</h3>
          <div class="form-group">
            <label>Ulica i broj</label>
            <input type="text" id="adresa">
          </div>
          <div class="form-row">
            <div class="form-group">
              <label>Grad</label>
              <input type="text" id="grad">
            </div>
            <div class="form-group">
              <label>Država</label>
              <input type="text" id="drzava">
            </div>
          </div>
        </div>
        
        <!-- Display Template -->
<div class="form-section">
  <h3>🎨 Dizajn vizitke</h3>
  <div class="template-grid">
        <label class="template-option">
      <input type="radio" name="display_template" value="business">
      <div class="template-preview" style="background: linear-gradient(135deg, #1e293b 0%, #334155 100%);"></div>
      <div>Business</div>
    </label>
  </div>
</div>

<!-- Accent Color -->
<div class="form-section">
  <h3>🎨 Glavna boja (Accent)</h3>
  <div class="form-row">
    <div class="form-group">
      <label>Odaberi boju</label>
      <div style="display: flex; gap: 10px; align-items: center;">
        <input type="color" id="accent_color" value="#3b82f6" style="width: 80px; height: 50px; cursor: pointer; border-radius: 8px; border: 2px solid #e5e7eb;">
        <input type="text" id="accent_color_hex" value="#3b82f6" placeholder="#3b82f6" style="flex: 1; font-family: monospace; text-transform: uppercase;" maxlength="7" pattern="^#[0-9A-Fa-f]{6}$">
        <button type="button" onclick="resetAccentColor()" style="padding: 12px 16px; background: #6b7280; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 600; white-space: nowrap;">↺ Default</button>
      </div>
      <small style="color: #64748b; font-size: 12px; margin-top: 8px; display: block;">Default boja: #3b82f6</small>
    </div>
    <div class="form-group">
      <label>Pozadinska slika headera (opciono)</label>
      <input type="file" id="background_image" accept="image/*">
      <small style="color: #64748b; font-size: 12px;">Slika će imati overlay efekt preko nje</small>
    </div>
  </div>
</div>
        
        <!-- Social links -->
        <div class="form-section">
          <h3>🔗 Društvene mreže</h3>
          <div id="socialLinksContainer" class="social-links-editor"></div>
          <button type="button" class="add-link-btn" onclick="addSocialLink()">+ Dodaj društvenu mrežu</button>
        </div>
        
        <div class="form-section">
  <h3>📊 Analytics pristup</h3>
  <div class="form-group">
    <label>Analytics ključ (za javni pristup)</label>
    <div style="display: flex; gap: 10px;">
      <input type="text" id="analytics_key" readonly placeholder="Generirat će se automatski" style="flex: 1; background: #f5f7fa; font-family: monospace;">
      <button type="button" class="btn btn-small" onclick="generateAnalyticsKey()" style="width: auto; white-space: nowrap;">🔄 Generiraj</button>
    </div>
    <small style="color: #64748b; font-size: 12px; margin-top: 5px; display: block;">
      Link za pristup: <span id="analyticsLinkPreview" style="color: #667eea; font-weight: 600;">-</span>
    </small>
  </div>
  
  <div class="form-group">
    <label>Analytics Lozinka</label>
    <div style="display: flex; gap: 10px;">
      <input type="text" id="analyticsPassword" placeholder="Automatski generirano" style="flex: 1;">
      <button type="button" onclick="generateNewPassword()" style="padding: 8px 15px; background: #667eea; color: white; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;">Nova</button>
    </div>
    <small style="color: #64748b; font-size: 12px; margin-top: 5px; display: block;">6-8 znakova (slova, brojevi, . ? -)</small>
  </div>
  
<div class="form-section">
  <h3>⏰ Aktivnost računa</h3>
  <div class="checkbox-item" style="margin-bottom: 15px;">
    <input type="checkbox" id="active" checked>
    <label>Račun je aktivan</label>
  </div>
  <div class="form-row">
    <div class="form-group">
      <label>Datum aktivacije (opciono)</label>
      <input type="date" id="active_from" value="">
    </div>
    <div class="form-group">
      <label>Aktivan do (prazno = lifetime)</label>
      <input type="date" id="active_until" value="">
      <button type="button" onclick="document.getElementById('active_until').value='';" style="padding: 8px; background: #6b7280; color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 12px; margin-top: 5px;">
        Ukloni datum
      </button>
    </div>
  </div>
</div>
</div>
        
        <!-- Moduli -->
        <div class="form-section">
          <h3>⚙️ Moduli</h3>
          <div class="checkbox-grid">
            <label class="checkbox-item">
              <input type="checkbox" id="module_vcard" checked>
              <span>💾 Spremi kontakt (vCard)</span>
            </label>
            <label class="checkbox-item">
              <input type="checkbox" id="module_qr_code" checked>
              <span>📱 QR kod</span>
            </label>
            <label class="checkbox-item">
              <input type="checkbox" id="module_mapa">
              <span>🗺️ Google Mapa</span>
            </label>
            <label class="checkbox-item">
              <input type="checkbox" id="module_instagram_feed">
              <span>📷 Instagram feed</span>
            </label>
            <label class="checkbox-item">
  <input type="checkbox" id="module_text_block">
  <span>📝 Tekstualni blok</span>
</label>

<div id="textBlockConfig" class="form-group" style="display: none; margin-top: 15px; grid-column: 1 / -1;">
  <label>Sadržaj tekstualnog bloka</label>
  <textarea id="text_block_content" rows="4" placeholder="Unesite tekst koji želite prikazati..."></textarea>
</div>

<label class="checkbox-item">
  <input type="checkbox" id="module_google_reviews">
  <span>⭐ Google ocjene</span>
</label>

<div id="googleReviewsConfig" class="form-group" style="display: none; margin-top: 15px; grid-column: 1 / -1;">
  <label>Google My Business link za recenzije</label>
  <input type="text" id="google_reviews_link" placeholder="Zalijepite cijeli Google link za recenzije">
  <small style="color: #64748b; font-size: 12px; margin-top: 5px; display: block;">
    Primjer: https://g.page/r/...  ili skraćeni link s Google Maps
  </small>
</div>

<label class="checkbox-item">
  <input type="checkbox" id="module_working_hours">
  <span>🕐 Radno vrijeme</span>
</label>

<div id="workingHoursConfig" class="form-group" style="display: none; margin-top: 15px; grid-column: 1 / -1;">
  <label>Tip radnog vremena</label>
  <select id="working_hours_type" onchange="updateWorkingHoursFields()">
    <option value="all_days">Isto za sve dane</option>
    <option value="weekday_weekend">Radni dani / Subota / Nedjelja</option>
  </select>
  
  <div id="allDaysFields" style="margin-top: 15px;">
    <label>Radno vrijeme (svi dani)</label>
    <input type="text" id="working_hours_all" placeholder="npr. 08:00 - 20:00">
  </div>
  
  <div id="splitDaysFields" style="display: none; margin-top: 15px;">
    <div class="form-row">
      <div class="form-group">
        <label>Ponedjeljak - Petak</label>
        <input type="text" id="working_hours_weekday" placeholder="npr. 08:00 - 20:00">
      </div>
      <div class="form-group">
        <label>Subota</label>
        <input type="text" id="working_hours_saturday" placeholder="npr. 09:00 - 15:00">
      </div>
    </div>
    <div class="form-group">
      <label>Nedjelja</label>
      <input type="text" id="working_hours_sunday" placeholder="npr. Zatvoreno ili 10:00 - 14:00">
    </div>
  </div>
</div>
          </div>
          
          <div id="mapaConfig" class="form-group" style="display: none; margin-top: 15px;">
            <label>Google Maps link ili naziv lokacije</label>
            <input type="text" id="mapa_link" placeholder="Npr: https://maps.app.goo.gl/... ili 'Ljubuški, BiH'">
          </div>
          
          <div id="instagramConfig" class="form-group" style="display: none; margin-top: 15px;">
            <label>Instagram korisničko ime</label>
            <input type="text" id="instagram_username" placeholder="Bez @, npr: zvonimirmadzar">
          </div>
          
          <!-- QR Code Preview -->
          <div id="qrCodePreview"></div>
        </div>
        
        <div style="display: flex; gap: 10px; margin-top: 30px;">
          <button type="submit" class="btn">💾 Spremi</button>
          <button type="button" class="btn btn-secondary" onclick="closeModal()">Odustani</button>
        </div>
      </form>
    </div>
  </div>

 <script>
 
 // Generiraj analytics password
function generateNewPassword() {
  const chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.?-';
  let password = '';
  const length = Math.floor(Math.random() * 3) + 6; // 6-8 znakova
  
  for (let i = 0; i < length; i++) {
    password += chars[Math.floor(Math.random() * chars.length)];
  }
  
  document.getElementById('analyticsPassword').value = password;
}
 
 // Generiraj analytics key
function generateAnalyticsKey() {
  const chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
  let key = '';
  for (let i = 0; i < 32; i++) {
    key += chars[Math.floor(Math.random() * chars.length)];
  }
  document.getElementById('analytics_key').value = key;
  
  // Generiraj i short code
  const shortCode = generateShortCode();
  if (!document.getElementById('analytics_short_code')) {
    const input = document.createElement('input');
    input.type = 'hidden';
    input.id = 'analytics_short_code';
    document.getElementById('userForm').appendChild(input);
  }
  document.getElementById('analytics_short_code').value = shortCode;
  
  updateAnalyticsLink();
}

function generateShortCode() {
  const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
  let code = '';
  for (let i = 0; i < 7; i++) {
    code += chars[Math.floor(Math.random() * chars.length)];
  }
  return code;
}

function updateAnalyticsLink() {
  const slug = document.getElementById('slug').value;
  const shortCode = document.getElementById('analytics_short_code')?.value;
  const preview = document.getElementById('analyticsLinkPreview');
  
  if (slug && shortCode) {
    preview.textContent = `https://card.taptap.ba/a/${slug}-${shortCode}`;
    preview.style.cursor = 'pointer';
    preview.onclick = () => {
      navigator.clipboard.writeText(`https://card.taptap.ba/a/${slug}-${shortCode}`);
      alert('Link kopiran!');
    };
  } else {
    preview.textContent = '-';
  }
}

 
 
    const API_URL = '/api';
    let currentToken = localStorage.getItem('adminToken');
    let editingUserId = null;
    let linkCounter = 0;

    if (currentToken) {
        document.getElementById('loginView').style.display = 'none';
        document.getElementById('dashboardView').style.display = 'block';
        loadUsers();
    }

    async function handleLogin() {
        const username = document.getElementById('loginUsername').value;
        const password = document.getElementById('loginPassword').value;
        
        try {
            const response = await fetch(`${API_URL}/auth/login`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ username, password })
            });
            
            const data = await response.json();
            
            if (response.ok) {
                currentToken = data.token;
                localStorage.setItem('adminToken', currentToken);
                document.getElementById('loginView').style.display = 'none';
                document.getElementById('dashboardView').style.display = 'block';
                loadUsers();
            } else {
                document.getElementById('loginError').textContent = data.error || 'Neispravni podaci';
                document.getElementById('loginError').style.display = 'block';
            }
        } catch (error) {
            document.getElementById('loginError').textContent = 'Greška pri prijavi';
            document.getElementById('loginError').style.display = 'block';
        }
    }

    function handleLogout() {
        localStorage.removeItem('adminToken');
        currentToken = null;
        document.getElementById('dashboardView').style.display = 'none';
        document.getElementById('loginView').style.display = 'flex';
    }
    
    function showTab(tabName) {
        // Update tab buttons
        document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
        event.target.classList.add('active');
        
        // Update tab content
        document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
        document.getElementById(`${tabName}-tab`).classList.add('active');
        
        // Load data if needed
        switch(tabName) {
            case 'users':
                loadUsers();
                break;
            case 'settings':
                loadSettings();
                break;
        }
    }

    async function loadUsers() {
        try {
            const response = await fetch(`${API_URL}/users`, {
                headers: { 'Authorization': `Bearer ${currentToken}` }
            });
            
            const users = await response.json();
            renderUsers(users);
        } catch (error) {
            console.error('Greška:', error);
        }
    }

    async function renderUsers(users) {
        const container = document.getElementById('usersList');
        
        let html = '';
        for (const user of users) {
            const analytics = await fetchAnalytics(user.id);
            
            html += `
                <div class="user-card">
                    <div class="user-header">
                        <div class="user-info">
                            <h3>${user.ime} ${user.prezime} <span style="color: #94a3b8; font-size: 14px; font-weight: 400;">(ID: ${user.id})</span></h3>
                            <div class="user-slug">
                                <a href="/${user.slug}" target="_blank">card.taptap.ba/${user.slug}</a>
                            </div>
                        </div>
                        <div class="user-actions">
  <button class="btn btn-small" onclick="editUser(${user.id})">✏️ Uredi</button>
  <button class="btn btn-small" style="background: #10b981;" onclick="exportUser(${user.id})">⬇️ Export</button>
  ${user.analytics_key ? `<button class="btn btn-small" style="background: #8b5cf6;" onclick="window.open('https://card.taptap.ba/${user.slug}/analytics/${user.analytics_key}', '_blank')">📊</button>` : ''}
  <button class="btn btn-small btn-danger" onclick="deleteUser(${user.id}, '${user.slug}')">🗑️</button>
</div>
                    </div>
                    <div class="stats">
                        <div class="stat-box">
                            <div class="stat-number">${analytics.total}</div>
                            <div class="stat-label">Ukupno pregleda</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number">${analytics.qr_scans}</div>
                            <div class="stat-label">QR skeniranja</div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        container.innerHTML = html;
    }

    async function fetchAnalytics(userId) {
        try {
            const response = await fetch(`${API_URL}/analytics/${userId}`, {
                headers: { 'Authorization': `Bearer ${currentToken}` }
            });
            return await response.json();
        } catch (error) {
            return { total: 0, qr_scans: 0 };
        }
    }

    function openCreateModal() {
        editingUserId = null;
        document.getElementById('modalTitle').textContent = 'Novi korisnik';
        document.getElementById('userForm').reset();
        document.getElementById('userId').value = '';
        document.getElementById('slug').disabled = false;
        document.getElementById('socialLinksContainer').innerHTML = '';
        document.getElementById('qrCodePreview').innerHTML = '';
        
        // Reset accent color na default
        const defaultColor = '#3b82f6';
        const accentColorInput = document.getElementById('accent_color');
        const accentColorHexInput = document.getElementById('accent_color_hex');
        
        if (accentColorInput) accentColorInput.value = defaultColor;
        if (accentColorHexInput) accentColorHexInput.value = defaultColor;
        
        document.getElementById('userModal').classList.add('active');
        
        // Default values
        const modernTemplate = document.querySelector('input[name="display_template"][value="modern"]');
        const purpleBlueBackground = document.querySelector('input[name="background"][value="purple-blue"]');
        
        if (modernTemplate) modernTemplate.checked = true;
        if (purpleBlueBackground) purpleBlueBackground.checked = true;
        
        updateTemplateSelection();
        
        // Inicijaliziraj accent color sinkronizaciju
        initAccentColorSync();
        
        // Auto-generiraj lozinku za novog korisnika
        generateNewPassword();
    }

    async function editUser(userId) {
        editingUserId = userId;
        document.getElementById('modalTitle').textContent = 'Uredi korisnika';
        
        try {
            const response = await fetch(`${API_URL}/users/${userId}`, {
                headers: { 'Authorization': `Bearer ${currentToken}` }
            });
            
            const user = await response.json();
            
            document.getElementById('userId').value = user.id;
            document.getElementById('slug').value = user.slug;
            document.getElementById('slug').disabled = true;
            document.getElementById('ime').value = user.ime;
            document.getElementById('prezime').value = user.prezime;
            document.getElementById('opis').value = user.opis || '';
            document.getElementById('email').value = user.email || '';
            document.getElementById('telefon').value = user.telefon || '';
            document.getElementById('whatsapp').value = user.whatsapp || '';
            document.getElementById('viber').value = user.viber || '';
            document.getElementById('adresa').value = user.adresa || '';
            document.getElementById('grad').value = user.grad || '';
            document.getElementById('drzava').value = user.drzava || '';
            
            // Template i accent color
            const templateRadio = document.querySelector(`input[name="display_template"][value="${user.display_template}"]`);
            if (templateRadio) {
                templateRadio.checked = true;
                updateTemplateSelection();
            }
            
            const accentColor = user.accent_color || '#3b82f6';
            document.getElementById('accent_color').value = accentColor;
            document.getElementById('accent_color_hex').value = accentColor;
            
            // Analytics key
            document.getElementById('analytics_key').value = user.analytics_key || '';
            if (user.analytics_short_code) {
                if (!document.getElementById('analytics_short_code')) {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.id = 'analytics_short_code';
                    document.getElementById('userForm').appendChild(input);
                }
                document.getElementById('analytics_short_code').value = user.analytics_short_code;
            }
            updateAnalyticsLink();
            
            // Aktivnost
            document.getElementById('active').checked = Boolean(user.active);
            document.getElementById('active_from').value = user.active_from || '';
            document.getElementById('active_until').value = user.active_until || '';
            
            // Social links
            document.getElementById('socialLinksContainer').innerHTML = '';
            (user.links || []).forEach(link => {
                addSocialLink(link.tip, link.url);
            });
            
            // Moduli
            (user.modules || []).forEach(module => {
                const checkbox = document.getElementById(`module_${module.modul_name}`);
                if (checkbox) {
                    checkbox.checked = Boolean(module.aktivan);
                    
                    // Config
                    if (module.config) {
                        const config = typeof module.config === 'string' ? JSON.parse(module.config) : module.config;
                        
                        if (module.modul_name === 'mapa' && config.link) {
                            document.getElementById('mapa_link').value = config.link;
                        }
                        if (module.modul_name === 'instagram_feed' && config.username) {
                            document.getElementById('instagram_username').value = config.username;
                        }
                        if (module.modul_name === 'text_block' && config.content) {
                            document.getElementById('text_block_content').value = config.content;
                        }
                        if (module.modul_name === 'google_reviews' && config.link) {
                            document.getElementById('google_reviews_link').value = config.link;
                        }
                        if (module.modul_name === 'working_hours') {
                            document.getElementById('working_hours_type').value = config.type || 'all_days';
                            updateWorkingHoursFields();
                            if (config.type === 'all_days' && config.all) {
                                document.getElementById('working_hours_all').value = config.all;
                            } else {
                                document.getElementById('working_hours_weekday').value = config.weekday || '';
                                document.getElementById('working_hours_saturday').value = config.saturday || '';
                                document.getElementById('working_hours_sunday').value = config.sunday || '';
                            }
                        }
                    }
                }
            });
            
            toggleModuleConfigs();
            
            document.getElementById('userModal').classList.add('active');
            initAccentColorSync();
            
        } catch (error) {
            console.error('Greška:', error);
        }
    }

    function closeModal() {
        document.getElementById('userModal').classList.remove('active');
        editingUserId = null;
    }

    function handleSubmit(e) {
        e.preventDefault();
        saveUser();
    }

    function addSocialLink(type = '', url = '') {
        const container = document.getElementById('socialLinksContainer');
        const id = linkCounter++;
        
        const linkHtml = `
            <div class="link-row" data-link-id="${id}">
                <select name="link_type_${id}">
                    <option value="website" ${type === 'website' ? 'selected' : ''}>Website</option>
                    <option value="facebook" ${type === 'facebook' ? 'selected' : ''}>Facebook</option>
                    <option value="instagram" ${type === 'instagram' ? 'selected' : ''}>Instagram</option>
                    <option value="linkedin" ${type === 'linkedin' ? 'selected' : ''}>LinkedIn</option>
                    <option value="twitter" ${type === 'twitter' ? 'selected' : ''}>Twitter</option>
                    <option value="youtube" ${type === 'youtube' ? 'selected' : ''}>YouTube</option>
                    <option value="tiktok" ${type === 'tiktok' ? 'selected' : ''}>TikTok</option>
                    <option value="github" ${type === 'github' ? 'selected' : ''}>GitHub</option>
                    <option value="custom" ${type === 'custom' ? 'selected' : ''}>Custom Link</option>
                </select>
                <input type="url" name="link_url_${id}" placeholder="https://..." value="${url}" required>
                <button type="button" class="remove-link-btn" onclick="this.parentElement.remove()">×</button>
            </div>
        `;
        
        container.insertAdjacentHTML('beforeend', linkHtml);
    }

    async function saveUser() {
        const formData = new FormData();
        
        if (editingUserId) {
            formData.append('_method', 'PUT');
            formData.append('user_id', editingUserId);
        }
        
        formData.append('slug', document.getElementById('slug').value);
        formData.append('ime', document.getElementById('ime').value);
        formData.append('prezime', document.getElementById('prezime').value);
        formData.append('opis', document.getElementById('opis').value);
        formData.append('email', document.getElementById('email').value);
        formData.append('telefon', document.getElementById('telefon').value);
        formData.append('whatsapp', document.getElementById('whatsapp').value);
        formData.append('viber', document.getElementById('viber').value);
        formData.append('adresa', document.getElementById('adresa').value);
        formData.append('grad', document.getElementById('grad').value);
        formData.append('drzava', document.getElementById('drzava').value);
        
        const slikaFile = document.getElementById('slika').files[0];
        if (slikaFile) {
            formData.append('slika', slikaFile);
        }
        
        const bgImageFile = document.getElementById('background_image').files[0];
        if (bgImageFile) {
            formData.append('background_image', bgImageFile);
        }
        
        const template = document.querySelector('input[name="display_template"]:checked');
        formData.append('display_template', template ? template.value : 'business');
        
        formData.append('accent_color', document.getElementById('accent_color').value);
        
        // Social links
        const linkRows = document.querySelectorAll('.link-row');
        const links = Array.from(linkRows).map((row, index) => {
            const id = row.dataset.linkId;
            return {
                tip: document.querySelector(`select[name="link_type_${id}"]`).value,
                url: document.querySelector(`input[name="link_url_${id}"]`).value,
                sort_order: index
            };
        });
        formData.append('links', JSON.stringify(links));
        
        // Moduli
        const modules = {
            vcard: document.getElementById('module_vcard').checked,
            qr_code: document.getElementById('module_qr_code').checked,
            mapa: document.getElementById('module_mapa').checked,
            instagram_feed: document.getElementById('module_instagram_feed').checked,
            text_block: document.getElementById('module_text_block')?.checked || false,
            google_reviews: document.getElementById('module_google_reviews')?.checked || false,
            working_hours: document.getElementById('module_working_hours')?.checked || false
        };
        
        const analyticsKey = document.getElementById('analytics_key').value;
        
        formData.append('active', document.getElementById('active').checked ? '1' : '0');
formData.append('active_from', document.getElementById('active_from').value);
formData.append('active_until', document.getElementById('active_until').value);

// Generiraj short code ako ne postoji
if (!document.getElementById('analytics_key').value) {
  generateAnalyticsKey();
}
const shortCode = document.getElementById('analytics_short_code')?.value || generateShortCode();
formData.append('analytics_short_code', shortCode);


if (analyticsKey) {
  formData.append('analytics_key', analyticsKey);
}

// Analytics password
const analyticsPassword = document.getElementById('analyticsPassword').value;
if (analyticsPassword) {
  formData.append('analytics_password', analyticsPassword);
}
        
        formData.append('modules', JSON.stringify(modules));
        
        // Module configs
        if (modules.mapa) {
            formData.append('mapa_link', document.getElementById('mapa_link').value);
        }
        if (modules.instagram_feed) {
            formData.append('instagram_username', document.getElementById('instagram_username').value);
        }
        
        // NOVI MODULI CONFIG
        if (modules.text_block) {
            formData.append('text_block_content', document.getElementById('text_block_content').value);
        }
        if (modules.google_reviews) {
            formData.append('google_reviews_link', document.getElementById('google_reviews_link').value);
        }
        if (modules.working_hours) {
            const whType = document.getElementById('working_hours_type').value;
            formData.append('working_hours_type', whType);
            
            if (whType === 'all_days') {
                formData.append('working_hours_all', document.getElementById('working_hours_all').value);
            } else {
                formData.append('working_hours_weekday', document.getElementById('working_hours_weekday').value);
                formData.append('working_hours_saturday', document.getElementById('working_hours_saturday').value);
                formData.append('working_hours_sunday', document.getElementById('working_hours_sunday').value);
            }
        }
        
        try {
            const url = editingUserId ? `${API_URL}/users/${editingUserId}` : `${API_URL}/users`;
            
            const response = await fetch(url, {
                method: 'POST',
                headers: { 'Authorization': `Bearer ${currentToken}` },
                body: formData
            });
            
            const responseData = await response.json();
            
            if (response.ok) {
                closeModal();
                loadUsers();
            } else {
                document.getElementById('modalError').textContent = responseData.error || 'Greška';
                document.getElementById('modalError').style.display = 'block';
            }
        } catch (error) {
            console.error('Submit error:', error);
            document.getElementById('modalError').textContent = 'Greška pri spremanju: ' + error.message;
            document.getElementById('modalError').style.display = 'block';
        }
    }

    async function deleteUser(userId, slug) {
        if (!confirm(`Jesi li siguran da želiš obrisati korisnika ${slug}?`)) return;
        
        try {
            await fetch(`${API_URL}/users/${userId}`, {
                method: 'DELETE',
                headers: { 'Authorization': `Bearer ${currentToken}` }
            });
            loadUsers();
        } catch (error) {
            alert('Greška pri brisanju');
        }
    }

    // Template i background selection highlighting
    document.querySelectorAll('.template-option').forEach(option => {
        option.addEventListener('click', updateTemplateSelection);
    });

    function updateTemplateSelection() {
        document.querySelectorAll('.template-option').forEach(opt => opt.classList.remove('active'));
        const selected = document.querySelector('input[name="display_template"]:checked');
        if (selected) {
            selected.parentElement.classList.add('active');
        }
    }

    // Module config toggling
    document.getElementById('module_mapa')?.addEventListener('change', toggleModuleConfigs);
    document.getElementById('module_instagram_feed')?.addEventListener('change', toggleModuleConfigs);
    document.getElementById('module_text_block')?.addEventListener('change', toggleModuleConfigs);
    document.getElementById('module_google_reviews')?.addEventListener('change', toggleModuleConfigs);
    document.getElementById('module_working_hours')?.addEventListener('change', toggleModuleConfigs);

    function toggleModuleConfigs() {
        document.getElementById('mapaConfig').style.display = 
            document.getElementById('module_mapa').checked ? 'block' : 'none';
        document.getElementById('instagramConfig').style.display = 
            document.getElementById('module_instagram_feed').checked ? 'block' : 'none';
        document.getElementById('textBlockConfig').style.display = 
            document.getElementById('module_text_block')?.checked ? 'block' : 'none';
        document.getElementById('googleReviewsConfig').style.display = 
            document.getElementById('module_google_reviews')?.checked ? 'block' : 'none';
        document.getElementById('workingHoursConfig').style.display = 
            document.getElementById('module_working_hours')?.checked ? 'block' : 'none';
    }

    // Nova funkcija za radno vrijeme
    function updateWorkingHoursFields() {
        const type = document.getElementById('working_hours_type').value;
        document.getElementById('allDaysFields').style.display = type === 'all_days' ? 'block' : 'none';
        document.getElementById('splitDaysFields').style.display = type === 'weekday_weekend' ? 'block' : 'none';
    }
    
    // Accent color sinkronizacija
    function initAccentColorSync() {
        const accentColorPicker = document.getElementById('accent_color');
        const accentColorHex = document.getElementById('accent_color_hex');
        
        if (!accentColorPicker || !accentColorHex) return;
        
        accentColorPicker.removeEventListener('input', accentColorPickerHandler);
        accentColorHex.removeEventListener('input', accentColorHexHandler);
        
        accentColorPicker.addEventListener('input', accentColorPickerHandler);
        accentColorHex.addEventListener('input', accentColorHexHandler);
    }
    
    function accentColorPickerHandler(e) {
        const accentColorHex = document.getElementById('accent_color_hex');
        if (accentColorHex) {
            accentColorHex.value = e.target.value.toUpperCase();
        }
    }
    
    function accentColorHexHandler(e) {
        const accentColorPicker = document.getElementById('accent_color');
        const value = e.target.value;
        if (/^#[0-9A-Fa-f]{6}$/.test(value) && accentColorPicker) {
            accentColorPicker.value = value;
        }
    }
    
    function resetAccentColor() {
        const defaultColor = '#3b82f6';
        const accentColorPicker = document.getElementById('accent_color');
        const accentColorHex = document.getElementById('accent_color_hex');
        
        if (accentColorPicker) accentColorPicker.value = defaultColor;
        if (accentColorHex) accentColorHex.value = defaultColor;
    }
    
    // Export pojedinačnog korisnika
async function exportUser(userId) {
  try {
    const response = await fetch(`${API_URL}/export/user/${userId}`, {
      headers: { 'Authorization': `Bearer ${currentToken}` }
    });
    
    const blob = await response.blob();
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `user-export-${userId}-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
  } catch (error) {
    alert('Greška pri exportu korisnika');
  }
}

// Export svih korisnika
async function exportAll() {
  if (!confirm('Želiš li exportati sve korisnike?')) return;
  
  try {
    const response = await fetch(`${API_URL}/export/all`, {
      headers: { 'Authorization': `Bearer ${currentToken}` }
    });
    
    const blob = await response.blob();
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `all-users-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
  } catch (error) {
    alert('Greška pri exportu korisnika');
  }
}

// Import korisnika
async function handleImport(event) {
  const file = event.target.files[0];
  if (!file) return;
  
  try {
    const text = await file.text();
    const data = JSON.parse(text);
    
    const response = await fetch(`${API_URL}/import`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${currentToken}`
      },
      body: JSON.stringify(data)
    });
    
    const result = await response.json();
    
    if (response.ok) {
      let message = `Uspješno importano ${result.imported} korisnika.`;
      if (result.errors.length > 0) {
        message += '\n\nGreške:\n' + result.errors.join('\n');
      }
      alert(message);
      loadUsers();
    } else {
      alert('Greška pri importu: ' + (result.error || 'Nepoznata greška'));
    }
  } catch (error) {
    alert('Greška pri čitanju datoteke. Provjeri je li format valjan JSON.');
  }
  
  // Reset input
  event.target.value = '';
}

// Settings funkcije
async function loadSettings() {
  try {
    const response = await fetch('/api/settings', {
      headers: { 'Authorization': `Bearer ${localStorage.getItem('adminToken')}` }
    });
    const settings = await response.json();
    
    document.getElementById('settingAnalyticsSlug').value = settings.analytics_slug || 'analitika';
    document.getElementById('previewSlug').textContent = settings.analytics_slug || 'analitika';
  } catch (error) {
    console.error('Greška:', error);
  }
}

async function saveSettings() {
  const analyticsSlug = document.getElementById('settingAnalyticsSlug').value.trim();
  
  if (!analyticsSlug) {
    alert('Analytics slug ne može biti prazan');
    return;
  }
  
  try {
    const response = await fetch('/api/settings', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
      },
      body: JSON.stringify({ analytics_slug: analyticsSlug })
    });
    
    const result = await response.json();
    
    if (result.success) {
      alert('✅ Postavke spremljene');
      loadSettings();
    } else {
      alert('Greška: ' + result.error);
    }
  } catch (error) {
    alert('Greška pri spremanju');
  }
}

// Auto-update preview
document.addEventListener('DOMContentLoaded', () => {
  const slugInput = document.getElementById('settingAnalyticsSlug');
  if (slugInput) {
    slugInput.addEventListener('input', (e) => {
      document.getElementById('previewSlug').textContent = e.target.value || 'analitika';
    });
  }
});


</script>
</body>
</html>